# -*- coding: binary -*-

require 'spec_helper'

RSpec.describe Rex::Proto::Kerberos::Model::TicketEncPart do
  subject(:ticket_enc_part) do
    described_class.new
  end

  let(:ticket_enc_part_sample) do
    "\x63\x82\x03\x2f\x30\x82\x03\x2b\xa0\x07\x03\x05\x00\x50\xa0\x00" \
    "\x00\xa1\x1b\x30\x19\xa0\x03\x02\x01\x17\xa1\x12\x04\x10\x4b\x55" \
    "\x4a\x51\x43\x73\x45\x52\x57\x55\x64\x49\x4b\x48\x6b\x78\xa2\x0a" \
    "\x1b\x08\x44\x57\x2e\x4c\x4f\x43\x41\x4c\xa3\x17\x30\x15\xa0\x03" \
    "\x02\x01\x01\xa1\x0e\x30\x0c\x1b\x0a\x66\x61\x6b\x65\x5f\x6d\x79" \
    "\x73\x71\x6c\xa4\x0b\x30\x09\xa0\x03\x02\x01\x00\xa1\x02\x04\x00" \
    "\xa5\x11\x18\x0f\x32\x30\x32\x32\x30\x36\x32\x30\x31\x30\x33\x39" \
    "\x34\x31\x5a\xa6\x11\x18\x0f\x32\x30\x32\x32\x30\x36\x32\x30\x31" \
    "\x30\x33\x39\x34\x31\x5a\xa7\x11\x18\x0f\x32\x30\x33\x32\x30\x36" \
    "\x31\x37\x31\x30\x33\x39\x34\x31\x5a\xa8\x11\x18\x0f\x32\x30\x33" \
    "\x32\x30\x36\x31\x37\x31\x30\x33\x39\x34\x31\x5a\xaa\x82\x02\x83" \
    "\x30\x82\x02\x7f\x30\x82\x02\x7b\xa0\x03\x02\x01\x01\xa1\x82\x02" \
    "\x72\x04\x82\x02\x6e\x30\x82\x02\x6a\x30\x82\x02\x66\xa0\x04\x02" \
    "\x02\x00\x80\xa1\x82\x02\x5c\x04\x82\x02\x58\x04\x00\x00\x00\x00" \
    "\x00\x00\x00\x01\x00\x00\x00\xbc\x01\x00\x00\x48\x00\x00\x00\x00" \
    "\x00\x00\x00\x0a\x00\x00\x00\x1e\x00\x00\x00\x08\x02\x00\x00\x00" \
    "\x00\x00\x00\x06\x00\x00\x00\x14\x00\x00\x00\x28\x02\x00\x00\x00" \
    "\x00\x00\x00\x07\x00\x00\x00\x14\x00\x00\x00\x40\x02\x00\x00\x00" \
    "\x00\x00\x00\x01\x10\x08\x00\xcc\xcc\xcc\xcc\xac\x01\x00\x00\xcc" \
    "\xcc\xcc\xcc\x6c\x5c\x00\x00\x80\x94\x11\x0c\x92\x84\xd8\x01\xff" \
    "\xff\xff\xff\xff\xff\xff\x7f\xff\xff\xff\xff\xff\xff\xff\x7f\x80" \
    "\x94\x11\x0c\x92\x84\xd8\x01\x00\x00\x00\x00\x00\x00\x00\x00\xff" \
    "\xff\xff\xff\xff\xff\xff\x7f\x14\x00\x14\x00\x97\xdf\x00\x00\x00" \
    "\x00\x00\x00\x60\xe2\x00\x00\x00\x00\x00\x00\x84\x9d\x00\x00\x00" \
    "\x00\x00\x00\x8c\x3b\x00\x00\x00\x00\x00\x00\x94\xc2\x00\x00\x00" \
    "\x00\x00\x00\x7e\xc9\x00\x00\xf4\x01\x00\x00\xf4\x01\x00\x00\x01" \
    "\x02\x00\x00\x05\x00\x00\x00\x72\x34\x00\x00\x00\x00\x00\x00\x00" \
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
    "\x00\x00\x00\x83\x88\x00\x00\x10\x00\x10\x00\x9c\x94\x00\x00\xa2" \
    "\xd7\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x02\x00\x00\x00" \
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0a" \
    "\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x66\x00\x61\x00\x6b" \
    "\x00\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00\x71\x00\x6c\x00\x00" \
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
    "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x01" \
    "\x02\x00\x00\x07\x00\x00\x00\x00\x02\x00\x00\x07\x00\x00\x00\x08" \
    "\x02\x00\x00\x07\x00\x00\x00\x06\x02\x00\x00\x07\x00\x00\x00\x07" \
    "\x02\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
    "\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00\x00\x44" \
    "\x00\x57\x00\x2e\x00\x4c\x00\x4f\x00\x43\x00\x41\x00\x4c\x00\x05" \
    "\x00\x00\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\x02" \
    "\x87\x5a\x55\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16\xf4\x01\x00\x00\x00" \
    "\x00\x00\x00\x80\x94\x11\x0c\x92\x84\xd8\x01\x14\x00\x66\x00\x61" \
    "\x00\x6b\x00\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00\x71\x00\x6c" \
    "\x00\x00\x00\x76\xff\xff\xff\xe2\x58\xd7\xf5\x06\x17\x64\x9c\x77" \
    "\xc6\xff\x16\x13\x16\x5d\xe3\x00\x00\x00\x00\x76\xff\xff\xff\xaf" \
    "\x64\xea\x1d\x9f\xa0\x63\xe9\xb1\x04\xd2\x53\xc4\x82\xce\x61\x00" \
    "\x00\x00\x00"
  end

  describe '#decode' do
    it 'decodes without errors' do
      ticket_enc_part.decode(ticket_enc_part_sample)
    end
    
    describe 'decoded ticket encrypted part' do
      subject(:ticket_enc_part_decoded) do
        ticket_enc_part.decode(ticket_enc_part_sample)
      end

      it 'decodes flags correctly' do
        expect(ticket_enc_part_decoded.flags).to eq(0x50a00000)

      end

      it 'decodes key correctly' do
        expect(ticket_enc_part_decoded.key.value).to eq("KUJQCsERWUdIKHkx")

      end

      it 'decodes crealm correctly' do
        expect(ticket_enc_part_decoded.crealm).to eq("DW.LOCAL")

      end

      it 'decodes cname correctly' do
        expect(ticket_enc_part_decoded.cname.name_string).to eq(["fake_mysql"])
      end

      it 'decodes transited correctly' do
        expect(ticket_enc_part_decoded.transited.contents).to eq("")
        expect(ticket_enc_part_decoded.transited.tr_type).to eq(0)
      end

      it 'decodes authtime correctly' do
        expect(ticket_enc_part_decoded.authtime).to eq("2022-06-20 10:39:41 UTC")
      end

      it 'decodes starttime correctly' do
        expect(ticket_enc_part_decoded.starttime).to eq("2022-06-20 10:39:41 UTC")
      end

      it 'decodes endtime correctly' do
        expect(ticket_enc_part_decoded.endtime).to eq("2032-06-17 10:39:41 UTC")
      end

      it 'decodes renew_till correctly' do
        expect(ticket_enc_part_decoded.renew_till).to eq("2032-06-17 10:39:41 UTC")
      end

      it 'decodes caddr correctly' do
        expect(ticket_enc_part.caddr).to eq(nil)
      end

      it 'decodes authorization_data correctly' do
        expect(ticket_enc_part.authorization_data).to eq(nil)
      end
    end
  end

  describe '#encode' do

    let(:ticket_flags) do
      Rex::Proto::Kerberos::Model::TicketFlags.new(0x50a00000)
    end

    let(:key) do
      encryption_key = Rex::Proto::Kerberos::Model::EncryptionKey.new
      encryption_key.type = 23
      encryption_key.value = "KUJQCsERWUdIKHkx"
      encryption_key
    end

    let(:cname) do
      principal_name = Rex::Proto::Kerberos::Model::PrincipalName.new
      principal_name.name_type = 1
      principal_name.name_string = ["fake_mysql"]
      principal_name
    end

    let(:transited) do
      transited_encoding = Rex::Proto::Kerberos::Model::TransitedEncoding.new
      transited_encoding.tr_type = 0
      transited_encoding.contents = ""
      transited_encoding
    end

    let(:sample_auth_data) do
      "\x30\x82\x02\x7f\x30\x82\x02\x7b\xa0\x03\x02\x01\x01\xa1\x82\x02" \
      "\x72\x04\x82\x02\x6e\x30\x82\x02\x6a\x30\x82\x02\x66\xa0\x04\x02" \
      "\x02\x00\x80\xa1\x82\x02\x5c\x04\x82\x02\x58\x04\x00\x00\x00\x00" \
      "\x00\x00\x00\x01\x00\x00\x00\xbc\x01\x00\x00\x48\x00\x00\x00\x00" \
      "\x00\x00\x00\x0a\x00\x00\x00\x1e\x00\x00\x00\x08\x02\x00\x00\x00" \
      "\x00\x00\x00\x06\x00\x00\x00\x14\x00\x00\x00\x28\x02\x00\x00\x00" \
      "\x00\x00\x00\x07\x00\x00\x00\x14\x00\x00\x00\x40\x02\x00\x00\x00" \
      "\x00\x00\x00\x01\x10\x08\x00\xcc\xcc\xcc\xcc\xac\x01\x00\x00\xcc" \
      "\xcc\xcc\xcc\x6c\x5c\x00\x00\x80\x94\x11\x0c\x92\x84\xd8\x01\xff" \
      "\xff\xff\xff\xff\xff\xff\x7f\xff\xff\xff\xff\xff\xff\xff\x7f\x80" \
      "\x94\x11\x0c\x92\x84\xd8\x01\x00\x00\x00\x00\x00\x00\x00\x00\xff" \
      "\xff\xff\xff\xff\xff\xff\x7f\x14\x00\x14\x00\x97\xdf\x00\x00\x00" \
      "\x00\x00\x00\x60\xe2\x00\x00\x00\x00\x00\x00\x84\x9d\x00\x00\x00" \
      "\x00\x00\x00\x8c\x3b\x00\x00\x00\x00\x00\x00\x94\xc2\x00\x00\x00" \
      "\x00\x00\x00\x7e\xc9\x00\x00\xf4\x01\x00\x00\xf4\x01\x00\x00\x01" \
      "\x02\x00\x00\x05\x00\x00\x00\x72\x34\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x83\x88\x00\x00\x10\x00\x10\x00\x9c\x94\x00\x00\xa2" \
      "\xd7\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x02\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0a" \
      "\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x66\x00\x61\x00\x6b" \
      "\x00\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00\x71\x00\x6c\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x01" \
      "\x02\x00\x00\x07\x00\x00\x00\x00\x02\x00\x00\x07\x00\x00\x00\x08" \
      "\x02\x00\x00\x07\x00\x00\x00\x06\x02\x00\x00\x07\x00\x00\x00\x07" \
      "\x02\x00\x00\x07\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" \
      "\x00\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x08\x00\x00\x00\x44" \
      "\x00\x57\x00\x2e\x00\x4c\x00\x4f\x00\x43\x00\x41\x00\x4c\x00\x05" \
      "\x00\x00\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\x02" \
      "\x87\x5a\x55\xfd\x66\x7b\xcc\xf7\xbc\xaf\x16\xf4\x01\x00\x00\x00" \
      "\x00\x00\x00\x80\x94\x11\x0c\x92\x84\xd8\x01\x14\x00\x66\x00\x61" \
      "\x00\x6b\x00\x65\x00\x5f\x00\x6d\x00\x79\x00\x73\x00\x71\x00\x6c" \
      "\x00\x00\x00\x76\xff\xff\xff\xe2\x58\xd7\xf5\x06\x17\x64\x9c\x77" \
      "\xc6\xff\x16\x13\x16\x5d\xe3\x00\x00\x00\x00\x76\xff\xff\xff\xaf" \
      "\x64\xea\x1d\x9f\xa0\x63\xe9\xb1\x04\xd2\x53\xc4\x82\xce\x61\x00" \
      "\x00\x00\x00"
    end

    let(:enc_part) do
      ticket_enc_part.flags = ticket_flags
      ticket_enc_part.key = key
      ticket_enc_part.crealm = "DW.LOCAL"
      ticket_enc_part.cname = cname
      ticket_enc_part.transited = transited
      ticket_enc_part.authtime = Time.parse("2022-06-20 10:39:41 UTC")
      ticket_enc_part.starttime = Time.parse("2022-06-20 10:39:41 UTC")
      ticket_enc_part.endtime = Time.parse("2032-06-17 10:39:41 UTC")
      ticket_enc_part.renew_till = Time.parse("2032-06-17 10:39:41 UTC")
      ticket_enc_part.authorization_data = Rex::Proto::Kerberos::Model::AuthorizationData.decode(sample_auth_data)
      ticket_enc_part
    end

    it 'encodes correctly' do
      expect(enc_part.encode).to eq(ticket_enc_part_sample)
    end
  end
end
