# frozen_string_literal: true

require 'spec_helper'
require 'msf/core/exploit/remote/ldap/active_directory'
require 'msf/core/exploit/remote/ldap/active_directory/security_descriptor_matcher'

RSpec.describe Msf::Exploit::Remote::LDAP::ActiveDirectory::SecurityDescriptorMatcher::Allow do
  let(:domain_sid) { 'S-1-5-32-12345-67890' }

  describe '.all' do
    let(:permissions) { %i[RP WP] }

    it 'returns a MultiAll instance' do
      expect(described_class.any(permissions)).to be_a Msf::Exploit::Remote::LDAP::ActiveDirectory::SecurityDescriptorMatcher::MultipleAll
    end

    it 'includes the described class as the sub-matchers' do
      expect(described_class.any(permissions).matchers).to all be_a described_class
    end
  end

  describe '.any' do
    let(:permissions) { %i[RP WP] }

    it 'returns a MultiAny instance' do
      expect(described_class.any(permissions)).to be_a Msf::Exploit::Remote::LDAP::ActiveDirectory::SecurityDescriptorMatcher::MultipleAny
    end

    it 'includes the described class as the sub-matchers' do
      expect(described_class.any(permissions).matchers).to all be_a described_class
    end
  end

  describe '.full_control' do
    let(:permissions) { %i[ CC DC LC SW RP WP DT LO CR SD RC WD WO ] }

    it 'returns an Allow instance' do
      expect(described_class.full_control).to be_a Msf::Exploit::Remote::LDAP::ActiveDirectory::SecurityDescriptorMatcher::Allow
    end

    it 'includes all of the permissions' do
      expect(described_class.full_control.permissions).to eq permissions
    end
  end

  describe '#apply_ace!' do
    let(:matcher) { described_class.new(%i[WP]) }

    context 'when the ace is ignored' do
      let(:ace) { Rex::Proto::MsDtyp::MsDtypAce.from_sddl_text('A;;RP;;;', domain_sid: domain_sid) }

      it 'checks if the ace should be ignored' do
        expect(matcher).to receive(:ignore_ace?).with(ace)
        matcher.apply_ace!(ace)
      end

      it 'does not apply the result' do
        allow(matcher).to receive(:ignore_ace?).with(ace).and_return(true)
        expect(matcher.apply_ace!(ace)).to be_nil
        expect(matcher.instance_variable_get(:@result)).to be_nil
      end
    end

    context 'when the ace is ignored' do
      let(:ace) { Rex::Proto::MsDtyp::MsDtypAce.from_sddl_text('A;;WP;;;', domain_sid: domain_sid) }

      it 'checks if the ace should be ignored' do
        expect(matcher).to receive(:ignore_ace?).with(ace)
        matcher.apply_ace!(ace)
      end

      it 'applies the result' do
        allow(matcher).to receive(:ignore_ace?).with(ace).and_return(false)
        expect(matcher.apply_ace!(ace)).to be_nil
        expect(matcher.instance_variable_get(:@result)).to eq ace.header.ace_type
      end
    end
  end

  describe '#ignore_ace?' do
    matcher_guid = SecureRandom.uuid
    other_guid = SecureRandom.uuid
    [
      [ 'AU', %w[],       nil,          nil,          true  ],
      [ 'AU', %w[],       matcher_guid, nil,          true  ],
      [ 'AU', %w[RP],     nil,          nil,          true  ],
      [ 'AU', %w[RP],     matcher_guid, nil,          true  ],
      [ 'AU', %w[WP],     nil,          nil,          true  ],
      [ 'AU', %w[WP],     matcher_guid, nil,          true  ],
      [ 'AU', %w[RP WP],  nil,          nil,          true  ],
      [ 'AU', %w[RP WP],  matcher_guid, nil,          true  ],
      [ 'OA', %w[],       nil,          other_guid,   true  ],
      [ 'OA', %w[],       matcher_guid, matcher_guid, true  ],
      [ 'OA', %w[],       matcher_guid, other_guid,   true  ],
      [ 'OA', %w[RP],     nil,          other_guid,   true  ],
      [ 'OA', %w[RP],     matcher_guid, matcher_guid, true  ],
      [ 'OA', %w[RP],     matcher_guid, other_guid,   true  ],
      [ 'OA', %w[WP],     nil,          other_guid,   true  ],
      [ 'OA', %w[WP],     matcher_guid, matcher_guid, false ],
      [ 'OA', %w[WP],     matcher_guid, other_guid,   true  ],
      [ 'OA', %w[RP WP],  nil,          other_guid,   true  ],
      [ 'OA', %w[RP WP],  matcher_guid, matcher_guid, false ],
      [ 'OA', %w[RP WP],  matcher_guid, other_guid,   true  ],
      [ 'OD', %w[],       nil,          other_guid,   true  ],
      [ 'OD', %w[],       matcher_guid, matcher_guid, true  ],
      [ 'OD', %w[],       matcher_guid, other_guid,   true  ],
      [ 'OD', %w[RP],     nil,          other_guid,   true  ],
      [ 'OD', %w[RP],     matcher_guid, matcher_guid, true  ],
      [ 'OD', %w[RP],     matcher_guid, other_guid,   true  ],
      [ 'OD', %w[WP],     nil,          other_guid,   true  ],
      [ 'OD', %w[WP],     matcher_guid, matcher_guid, false ],
      [ 'OD', %w[WP],     matcher_guid, other_guid,   true  ],
      [ 'OD', %w[RP WP],  nil,          other_guid,   true  ],
      [ 'OD', %w[RP WP],  matcher_guid, matcher_guid, false ],
      [ 'OD', %w[RP WP],  matcher_guid, other_guid,   true  ],
      [ 'A',  %w[],       nil,          nil,          true  ],
      [ 'A',  %w[],       matcher_guid, nil,          true  ],
      [ 'A',  %w[RP],     nil,          nil,          true  ],
      [ 'A',  %w[RP],     matcher_guid, nil,          true  ],
      [ 'A',  %w[WP],     nil,          nil,          false ],
      [ 'A',  %w[WP],     matcher_guid, nil,          true  ],
      [ 'A',  %w[RP WP],  nil,          nil,          false ],
      [ 'A',  %w[RP WP],  matcher_guid, nil,          true  ],
      [ 'D',  %w[],       nil,          nil,          true  ],
      [ 'D',  %w[],       matcher_guid, nil,          true  ],
      [ 'D',  %w[RP],     nil,          nil,          true  ],
      [ 'D',  %w[RP],     matcher_guid, nil,          true  ],
      [ 'D',  %w[WP],     nil,          nil,          false ],
      [ 'D',  %w[WP],     matcher_guid, nil,          true  ],
      [ 'D',  %w[RP WP],  nil,          nil,          false ],
      [ 'D',  %w[RP WP],  matcher_guid, nil,          true  ],
    ].each.with_index(__LINE__ - 48) do |(ace_type, permissions, matcher_object_id, ace_object_id, expected), line|
      it "returns #{expected} for #{ace_type} with #{ace_object_id || 'none'} and #{permissions.join(', ')} permissions (line: #{line})" do
        matcher = described_class.new(%i[WP], object_id: matcher_object_id)
        ace_sddl_text = "#{ace_type};;#{permissions.join('')};#{ace_object_id};;"
        ace = Rex::Proto::MsDtyp::MsDtypAce.from_sddl_text(ace_sddl_text, domain_sid: domain_sid)
        expect(matcher.ignore_ace?(ace)).to be expected
      end
    end
  end

  context 'when testing the flow' do
    it 'is satisfied after a matching ACE is applied' do
      ace = Rex::Proto::MsDtyp::MsDtypAce.from_sddl_text('A;;WP;;;', domain_sid: domain_sid)
      matcher = described_class.new(%i[WP])

      expect(matcher.satisfied?).to be_falsey
      expect(matcher.ignore_ace?(ace)).to be_falsey
      expect(matcher.apply_ace!(ace)).to be_nil
      expect(matcher.satisfied?).to be_truthy
    end

    it 'is not satisfied after a non-matching ACE is applied' do
      ace = Rex::Proto::MsDtyp::MsDtypAce.from_sddl_text('A;;RP;;;', domain_sid: domain_sid)
      matcher = described_class.new(%i[WP])

      expect(matcher.satisfied?).to be_falsey
      expect(matcher.ignore_ace?(ace)).to be_truthy
      expect(matcher.apply_ace!(ace)).to be_nil
      expect(matcher.satisfied?).to be_falsey
    end
  end
end
