# frozen_string_literal: true

require 'yaml'

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class ResponsibilityPatternsConfig
        ConfigError = Class.new(StandardError)

        attr_reader :responsible_patterns, :dependent_patterns

        # @param file_path [String] Path to YAML config file
        # @raise [ConfigError] if file cannot be loaded or parsed
        def initialize(file_path)
          @file_path = file_path
          @config = load_config
          @responsible_patterns = parse_patterns('responsible')
          @dependent_patterns = parse_patterns('dependent')
        end

        private

        def load_config
          YAML.load_file(@file_path)
        rescue Errno::ENOENT
          raise ConfigError, "Config file not found: #{@file_path}"
        rescue Psych::SyntaxError => e
          raise ConfigError, "Invalid YAML syntax in #{@file_path}: #{e.message}"
        end

        def parse_patterns(key)
          patterns = @config[key]

          raise ConfigError, "Missing or invalid '#{key}' key in #{@file_path}. Expected an array of patterns." unless patterns.is_a?(Array)

          patterns.map do |pattern|
            Regexp.new(pattern)
          rescue RegexpError => e
            raise ConfigError, "Invalid regex pattern '#{pattern}' in #{@file_path}: #{e.message}"
          end
        end
      end
    end
  end
end
